/*
 * Fabric processing routines
 */
#include "libfma.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#include "lf_process_fabric.h"

/*
 * Local prototypes
 */
static int lf_alloc_processing_space(struct lf_fabric *fp);
static void lf_free_processing_space(struct lf_fabric *fp);

/*
 * First, allocate some extra space in the user section of xbars
 * ands NICs, saving away whatever might be there already/
 *
 * Then:
 *   label clos levels of xbars
 *   assign link verifies
 *   compute subfabric membership
 *   assign map distribution
 */
int
lf_process_fabric_for_topo(
  struct lf_fabric *fp)
{
  int alloced;
  int rc;

  alloced = FALSE;

  /* Attach some scratch memory to each xbar and NIC for us to use */
  rc = lf_alloc_processing_space(fp);
  if (rc == -1) LF_ERROR(("Error allocating scratch processing space"));
  alloced = TRUE;

  /* generate clos labels */
  rc = lf_clos_label_xbars(fp);
  if (rc == -1) goto except;

  /* assign link verifies */
  lf_assign_link_verifies(fp);

  /* figure out what subfabric each NIC port is on */
  lf_assign_subfabrics(fp);

  /* make map distribution assignments */
  lf_assign_map_distribution(fp);

 except:
  if (alloced) lf_free_processing_space(fp);
  return rc;
}

/*
 * Allocate space for working variables
 */
static int
lf_alloc_processing_space(
  struct lf_fabric *fp)
{
  struct lf_fp_xbar *fpxp;
  struct lf_fp_nic *fpnicp;
  struct lf_xbar *xp;
  struct lf_nic *nicp;
  struct lf_host *hp;
  int num_nics;
  int x;
  int h;
  int ni;
  int n;

  fpxp = NULL;
  fpnicp = NULL;

  /*
   * Assign all xbar structs, if any xbars
   */
  if (fp->num_xbars > 0) {
    /* allocate data space (all are in one alloced array for efficiency) */
    LF_CALLOC(fpxp, struct lf_fp_xbar, fp->num_xbars);

    for (x=0; x<fp->num_xbars; ++x) {
      xp = fp->xbars[x];
      fpxp[x].old_user = xp->user.v;
      xp->user.v = fpxp + x;
    }
  }

  /* first, count all the nics */
  num_nics = 0;
  for (h=0; h<fp->num_hosts; ++h) {
    hp = fp->hosts[h];
    num_nics += hp->num_nics;
  }
    
  /* If any NICs found, assign all NIC structs */
  if (num_nics > 0) {
    LF_CALLOC(fpnicp, struct lf_fp_nic, num_nics);

    ni=0;
    for (h=0; h<fp->num_hosts; ++h) {
      hp = fp->hosts[h];

      for (n=0; n<hp->num_nics; ++n) {
	nicp = hp->nics[n];
	fpnicp[ni].old_user = nicp->user.v;
	nicp->user.v = fpnicp + ni;
	LF_CALLOC(fpnicp[ni].subfabric_id, int, nicp->num_ports);
	++ni;
      }
    }
  }

  return 0;

 except:
  LF_FREE(fpxp);
  LF_FREE(fpnicp);
  return -1;
}

/*
 * Free temporary working space
 */
static void
lf_free_processing_space(
  struct lf_fabric *fp)
{
  struct lf_fp_xbar *fpxp;
  struct lf_fp_nic *fpnicp;
  struct lf_xbar *xp;
  struct lf_host *hp;
  struct lf_nic *nicp;
  int ni;
  int n;
  int x;
  int h;

  /* Free all xbar structs */
  if (fp->num_xbars > 0) {
    fpxp = fp->xbars[0]->user.v;
    for (x=0; x<fp->num_xbars; ++x) {
      xp = fp->xbars[x];
      xp->user.v = fpxp[x].old_user;
    }
    LF_FREE(fpxp);
  }

  /* Free NIC structs */
  fpnicp = NULL;
  ni = 0;
  for (h=0; h<fp->num_hosts; ++h) {
    hp = fp->hosts[h];

    if (hp->num_nics > 0) {

      /* The first NIC is the start of our allocated struct */
      if (fpnicp == NULL) {
	fpnicp = hp->nics[0]->user.v;
      }

      for (n=0; n<hp->num_nics; ++n) {
	nicp = hp->nics[n];
	LF_FREE(FP_NIC(nicp)->subfabric_id);
	nicp->user.v = fpnicp[ni].old_user;
	++ni;
      }
    }
  }
  LF_FREE(fpnicp);
}
